package com.vmware.samples.chassisRackVSphere;

import java.lang.reflect.Array;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.vmware.samples.chassisRackVSphere.model.Chassis;
import com.vmware.samples.chassisRackVSphere.model.ChassisInfo;
import com.vmware.samples.chassisRackVSphere.model.ModelObject;
import com.vmware.samples.chassisRackVSphere.model.Rack;
import com.vmware.vise.vim.data.VimObjectReferenceService;

/**
 * Simplified data store for the Chassis objects, and related utilities.
 *
 * ***************************************************************************** *
 * IMPORTANT: this implementation uses in-memory data to keep the setup easy,    *
 * a real implementation should retrieve data from a remote server or database   *
 * and have very minimal or no storing/caching in the java layer because the     *
 * server must remain stateless and be able to scale out.                        *
 * ***************************************************************************** *
 *
 * Note that this class is thread-safe but doesn't deal with complex operations
 * or large data sets. It is not intended to be used as-is!
 */
public class ObjectStore {

   // You can increase CHASSIS_COUNT to test paging/sorting
   private static final int CHASSIS_COUNT = 10;
   private static final int RACK_COUNT = 3;

   // The custom object types
   private static final String CHASSIS_TYPE = ChassisRackVSphereDataAdapter.CHASSIS_TYPE;
   private static final String RACK_TYPE = ChassisRackVSphereDataAdapter.RACK_TYPE;

   // Map of chassis objects used in the sample.
   // The key is the object's uid and the value the Chassis instance.
   private final Map<String, Chassis> _chassisMap =
         new HashMap<String, Chassis>(CHASSIS_COUNT);

   // Map of chassis objects used in the sample.
   // The key is the object's uid and the value the Rack instance.
   private final Map<String, Rack> _rackMap =
         new HashMap<String, Rack>(RACK_COUNT);

   // Internal index used to create unique ids
   private static int _index = 0;

   private final VimObjectReferenceService _vimObjectRefService;

   // Resolver for URIs of resource objects.
   private final ModelObjectUriResolver _uriResolver;

   /**
    * Helper method to enforce name uniqueness.
    *
    * @param map  The object map passed in a thread-safe manner.
    * @param name The name to check
    * @return true if the given chassis name is already taken.
    */
   private static boolean mapContainsName(Map<String, Chassis> map, String name) {
      Iterator<String> i = map.keySet().iterator();
      while(i.hasNext()) {
         String uid = i.next();
         Chassis chassis = map.get(uid);
         if (name.equals(chassis.getName())) {
            return true;
         }
      }
      return false;
   }

   /**
    * Constructor.
    *
    * @param uriResolver
    *    Custom type resolver for ModelObject resources used in this sample
    *
    * @param vimObjectRefService
    *    Internal VimObjectReferenceService to get references of host objects
    *    references because this simple implementation doesn't worry about namespace.
    *
    * @see com.vmware.vise.vim.data.VimObjectReferenceService
    */
   public ObjectStore(
            ModelObjectUriResolver uriResolver,
            VimObjectReferenceService vimObjectRefService) {
      _uriResolver = uriResolver;
      _vimObjectRefService = vimObjectRefService;
   }

   /**
    * Initializes the local object store with Chassis and Rack objects.
    * This bean init method is defined in bundle-context.xml.
    *
    * Notes:
    * 1) This initialization done during bean creation must not be time-consuming.
    *  A real implementation will need some info to access its back-end. It can be stored
    *  in a properties file, or fetched from the plugin's vCenter extension meta data,
    *  or else some configuration will be required in the UI.
    * 2) Host relations cannot be added to chassis here because we don't have access to hosts
    *  data before the user logs in. See ChassisRackVSphereDataAdapter.initializeHostRelations.
    */
   public void init() {

      Rack[] allRacks = new Rack[RACK_COUNT];
      for (int i = 0; i < RACK_COUNT; i++) {
         allRacks[i] = createRack("Rack "+(i+1), "37.40, -122.14");
      }

      Chassis[] allChassis = new Chassis[CHASSIS_COUNT];
      for (int i = 0; i < CHASSIS_COUNT; i++) {
         String[] data = {"Chassis "+(i+1), "Server_Type "+ (i%3), "20in x 30in x 17in"};
         // The first 2 racks are associated with even or odd chassis numbers
         // (other racks are left empty and use a custom icon and label)
         Rack relatedRack = allRacks[i % 2];

         allChassis[i] = createChassis(new ChassisInfo(data), relatedRack, false);
      }
   }

   /**
    * Bean destroy method defined in bundle-context.xml.
    */
   public void destroy() {
      // nothing to clean-up in this sample.
   }

   /**
    * Get the current chassis objects in a thread-safe manner which allows the caller
    * to iterate through the list. Making a copy and exposing ObjectStore internals is
    * OK for this small example. A real implementation should provide a thread-safe
    * way of iterating through objects stored in a back-end server, without caching
    * them in memory.
    *
    * @return a copy of the current map of chassis.
    */
   Map<String, ModelObject> getAllChassis() {
      synchronized(_chassisMap) {
         return new HashMap<String, ModelObject>(_chassisMap) ;
      }
   }

   /**
    * @return a copy of the current map of racks.
    */
   Map<String, ModelObject> getAllRacks() {
      synchronized(_rackMap) {
         return new HashMap<String, ModelObject>(_rackMap) ;
      }
   }

   /**
    * Access a Chassis or Rack object.
    *
    * @param uri the object URI
    * @return the ModelObject or null if none was found.
    */
   ModelObject getModelObject(URI uri) {
      String type = _uriResolver.getResourceType(uri);
      String uid = _uriResolver.getUid(uri);

      if (CHASSIS_TYPE.equals(type)) {
         return getChassis(uid);
      } else if (RACK_TYPE.equals(type)) {
         return getRack(uid);
      }
      return null;
   }

   /**
    * Access a Chassis from the internal object store.
    *
    * @param uid  Unique identifier.
    * @return the Chassis object for the given uid, or null.
    */
   Chassis getChassis(String uid) {
      synchronized(_chassisMap) {
         return _chassisMap.get(uid);
      }
   }

   /**
    * Access a Rack from the internal object store.
    *
    * @param uid  Unique identifier.
    * @return the Rack object for the given uid, or null.
    */
   Rack getRack(String uid) {
      synchronized(_rackMap) {
         return _rackMap.get(uid);
      }
   }

   /**
    * Remove a Chassis from the internal object store.
    *
    * @param uid  Unique identifier.
    * @return the Chassis object that was removed, or null if no object was found for that uid.
    */
   Chassis removeChassis(String uid) {
      synchronized(_chassisMap) {
         Chassis chassis = _chassisMap.remove(uid);

         // Remove the rack-chassis relation if any
         if (chassis != null) {
            Rack rack = chassis.getRack();
            if (rack != null) {
               rack.removeChassis(chassis);
            }
         }
         return chassis;
      }
   }

   /**
    * Add a new Chassis to the object store
    *
    * @param chassisInfo
    *          The data used to create the chassis.
    * @param rack
    *          The Rack to associate with this chassis, or null when a chassis is
    *          created with no relation to a rack.
    * @param checkName
    *          true if the name should be checked first, to ensure unique names.
    * @return
    *          The new Chassis object, or null if the name is already taken.
    */
   Chassis createChassis(ChassisInfo chassisInfo, Rack rack, boolean checkName) {
      synchronized(_chassisMap) {
         // Enforce unique names
         if (checkName && mapContainsName(_chassisMap, chassisInfo.name)) {
            return null;
         }

         // Resource id that would normally be supplied by the back-end.
         String id = _uriResolver.createResourceId("server1", "chassis-"+(_index++));

         Chassis chassis = new Chassis(chassisInfo, rack, id);

         if (rack != null) {
            // Create the relation rack->chassis
            rack.addChassis(chassis);
         }

         // Add chassis to object store. Use the uid as key instead of resource id for
         // convenience. Real implementation would store object in separate back-end.
         URI chassisRef = chassis.getUri(_uriResolver);
         String uid = _uriResolver.getUid(chassisRef);
         _chassisMap.put(uid, chassis);

         return chassis;
      }
   }

   /**
    * Replace an existing chassis entry with a new one sharing
    * the same uid.
    *
    * @param uid The chassis unique identifier.
    * @param newChassisInfo The data about the new chassis.
    * @return true if success, false if the uid didn't exist anymore
    */
   boolean replaceChassis(String uid, ChassisInfo newChassisInfo) {
      synchronized(_chassisMap) {
         if (!_chassisMap.containsKey(uid)) {
            return false;
         }

         Chassis oldChassis = _chassisMap.get(uid);
         Rack rack = oldChassis.getRack();
         String chassisId = oldChassis.getId();

         // Create a new chassis re-using the same resource id, so the uid is also the same
         Chassis newChassis = new Chassis(newChassisInfo, rack, chassisId);
         assert(uid.equals(_uriResolver.getUid(newChassis.getUri(_uriResolver))));

         _chassisMap.put(uid, newChassis);

         if (rack != null) {
            // Replace the old rack relation
            rack.replaceChassis(oldChassis, newChassis);
         }

         // Put the same hosts back on the new chassis
         Object[] hosts = oldChassis.getHosts();
         for (int i = 0; i < hosts.length; i++) {
            addHostToChassis(hosts[i], uid);
         }
         return true;
      }
   }

   /**
    * @param hostRef
    * @param chassisUid
    * @return true if success, false is failure.
    */
   boolean addHostToChassis(Object hostRef, String chassisUid) {
      synchronized(_chassisMap) {
         Chassis chassis = _chassisMap.get(chassisUid);
         if (chassis == null) {
            return false;
         }

         // Remove the host from another chassis if found there
         for (Chassis ch: _chassisMap.values()) {
            if (ch == chassis) {
               continue;
            }
            if (ch.removeHost(hostRef, _vimObjectRefService)) {
               break;
            }
         }
         return chassis.addHost(hostRef, _vimObjectRefService);
      }
   }

   private Rack createRack(String name, String location) {
      synchronized(_rackMap) {

         // Resource id that would normally be supplied by the back-end.
         String id = _uriResolver.createResourceId("server1", "rack-"+(_index++));

         Rack rack = new Rack(name, location, id);

         // Add rack to object store. Use the uid as key instead of resource id for
         // convenience. Real implementation would store object in separate back-end.
         URI rackRef = rack.getUri(_uriResolver);
         String uid = _uriResolver.getUid(rackRef);
         _rackMap.put(uid, rack);

         return rack;
      }
   }

   public List<Object> getObjectsByPropertyValue(
            String targetType, String property, String comparableValue) {
      List<Object> objectList = new ArrayList<Object>();

      if (CHASSIS_TYPE.equals(targetType)) {
         synchronized(_chassisMap) {
            for (Chassis chassis : _chassisMap.values()) {
               Object propertyValue = chassis.getProperty(property);
               if (!(propertyValue instanceof String)) {
                  continue;
               }
               String lowerCaseVal = ((String)propertyValue).toLowerCase();
               if (lowerCaseVal.indexOf(comparableValue) >= 0) {
                  objectList.add(chassis);
               }
            }
         }
      } else if (RACK_TYPE.equals(targetType)) {
         synchronized(_rackMap) {
            for (Rack rack : _rackMap.values()) {
               Object propertyValue = rack.getProperty(property);
               if (!(propertyValue instanceof String)) {
                  continue;
               }
               String lowerCaseVal = ((String)propertyValue).toLowerCase();
               if (lowerCaseVal.indexOf(comparableValue) >= 0) {
                  objectList.add(rack);
               }
            }
         }
      }

      return objectList;
   }

   /**
    * Returns the chassis or rack containing a relation matching objectUid.
    */
   public ModelObject getRelationSource(
            String relation, String objectUid, String sourceType) {
      assert (relation != null && objectUid != null && sourceType != null);
      if (CHASSIS_TYPE.equals(sourceType)) {
         synchronized(_chassisMap) {
            for (Chassis chassis : _chassisMap.values()) {
               Object relationSource = chassis.getProperty(relation);
               if (isOrContains(relationSource, objectUid)) {
                  return chassis;
               }
            }
         }
      } else if (RACK_TYPE.equals(sourceType)) {
         synchronized(_rackMap) {
            for (Rack rack : _rackMap.values()) {
               Object relationSource = rack.getProperty(relation);
               if (isOrContains(relationSource, objectUid)) {
                  return rack;
               }
            }
         }
      }
      return null;
   }

   /**
    * Checks if the source and objectUid represent the same object or if
    * the source is an array then looks for the objectUid in the array.
    */
   private boolean isOrContains(Object source, String objectUid) {
      assert (source != null);
      if (!source.getClass().isArray()) {
         return representSameObject(source, objectUid);
      }
      int length = Array.getLength(source);
      for (int index = 0; index < length; ++index) {
         Object relatedObject = Array.get(source, index);
         if (representSameObject(relatedObject, objectUid)) {
            return true;
         }
      }
      return false;
   }

   /**
    * Check if objectUid and source object represent the same resource.
    */
   private boolean representSameObject(Object source, String objectUid) {
      assert (source != null);
      String uid = _vimObjectRefService.getUid(source);
      if (uid != null) {
         return uid.equals(objectUid);
      }
      return false;
   }

}
